import { useState } from 'react'
import { useNavigate } from 'react-router-dom'
import { motion, AnimatePresence } from 'framer-motion'
import { Headphones, Calendar, Check, ArrowRight, Piano, Guitar, Mic, Monitor, Minus } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './Rent.css'

const Rent = () => {
  const navigate = useNavigate()
  const [selectedCategory, setSelectedCategory] = useState('')
  const [selectedDate, setSelectedDate] = useState('')
  const [isSubmitted, setIsSubmitted] = useState(false)

  const categories = [
    { 
      id: 'keyboards', 
      name: 'Клавишные', 
      icon: Piano,
      description: 'Синтезаторы, пианино, MIDI-клавиатуры'
    },
    { 
      id: 'strings', 
      name: 'Струнные', 
      icon: Guitar,
      description: 'Гитары, бас-гитары, скрипки, виолончели'
    },
    { 
      id: 'microphones', 
      name: 'Микрофоны', 
      icon: Mic,
      description: 'Вокальные, инструментальные, студийные'
    },
    { 
      id: 'monitors', 
      name: 'Мониторы', 
      icon: Monitor,
      description: 'Студийные мониторы, PA-системы, колонки'
    }
  ]

  const today = new Date()
  const maxDate = new Date(today.getTime() + 180 * 24 * 60 * 60 * 1000)
  const minDateStr = today.toISOString().split('T')[0]
  const maxDateStr = maxDate.toISOString().split('T')[0]

  const handleSubmit = (e) => {
    e.preventDefault()
    if (!selectedCategory || !selectedDate) return

    const saved = localStorage.getItem('musicRentals')
    const rentals = saved ? JSON.parse(saved) : []
    
    const categoryData = categories.find(c => c.id === selectedCategory)
    
    const newRental = {
      id: Date.now(),
      equipment: `Оборудование: ${categoryData.name}`,
      date: selectedDate,
      review: null,
      photo: '/images/rental-placeholder.jpg'
    }
    
    rentals.push(newRental)
    localStorage.setItem('musicRentals', JSON.stringify(rentals))
    
    setIsSubmitted(true)
    setTimeout(() => {
      navigate('/')
    }, 2000)
  }

  return (
    <div className="rent-minimal-page">
      <div className="container">
        <motion.section 
          className="rent-header-minimal"
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          transition={{ duration: 0.6 }}
        >
          <div className="rent-header-content-minimal">
            <motion.div 
              className="rent-number-minimal"
              initial={{ opacity: 0, x: -30 }}
              animate={{ opacity: 1, x: 0 }}
              transition={{ delay: 0.2 }}
            >
              02
            </motion.div>
            <div className="rent-text-minimal">
              <motion.h1 
                className="rent-title-minimal"
                initial={{ opacity: 0, y: 20 }}
                animate={{ opacity: 1, y: 0 }}
                transition={{ delay: 0.3 }}
              >
                Арендовать оборудование
              </motion.h1>
              <motion.p 
                className="rent-description-minimal"
                initial={{ opacity: 0, y: 20 }}
                animate={{ opacity: 1, y: 0 }}
                transition={{ delay: 0.4 }}
              >
                Выберите категорию и дату возврата
              </motion.p>
            </div>
          </div>
        </motion.section>

        <div className="rent-layout-minimal">
          <motion.section 
            className="form-section-minimal"
            initial={{ opacity: 0, y: 30 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ delay: 0.5 }}
          >
            <form className="form-minimal" onSubmit={handleSubmit}>
              <div className="form-group-minimal">
                <label className="form-label-minimal">
                  <Headphones size={18} strokeWidth={1.5} />
                  <span>Категория</span>
                </label>
                <div className="categories-list-minimal">
                  {categories.map((category, index) => {
                    const Icon = category.icon
                    return (
                      <motion.button
                        key={category.id}
                        type="button"
                        className={`category-item-minimal ${selectedCategory === category.id ? 'selected' : ''}`}
                        onClick={() => setSelectedCategory(category.id)}
                        initial={{ opacity: 0, x: -20 }}
                        animate={{ opacity: 1, x: 0 }}
                        transition={{ delay: 0.6 + index * 0.1 }}
                        whileHover={{ x: 5 }}
                        whileTap={{ scale: 0.98 }}
                      >
                        <div className="category-icon-minimal">
                          <Icon size={20} strokeWidth={1.5} />
                        </div>
                        <div className="category-content-minimal">
                          <h3 className="category-name-minimal">{category.name}</h3>
                          <p className="category-desc-minimal">{category.description}</p>
                        </div>
                        {selectedCategory === category.id && (
                          <motion.div 
                            className="check-minimal"
                            initial={{ scale: 0 }}
                            animate={{ scale: 1 }}
                          >
                            <Check size={18} strokeWidth={2} />
                          </motion.div>
                        )}
                      </motion.button>
                    )
                  })}
                </div>
              </div>

              <AnimatePresence>
                {selectedCategory && (
                  <motion.div
                    className="form-group-minimal date-group-minimal"
                    initial={{ opacity: 0, height: 0 }}
                    animate={{ opacity: 1, height: 'auto' }}
                    exit={{ opacity: 0, height: 0 }}
                    transition={{ duration: 0.3 }}
                  >
                    <label className="form-label-minimal">
                      <Calendar size={18} strokeWidth={1.5} />
                      <span>Дата возврата</span>
                    </label>
                    <div className="date-input-wrapper-minimal">
                      <input
                        type="date"
                        value={selectedDate}
                        onChange={(e) => setSelectedDate(e.target.value)}
                        min={minDateStr}
                        max={maxDateStr}
                        className="date-input-minimal"
                        required
                      />
                    </div>
                  </motion.div>
                )}
              </AnimatePresence>

              <motion.button 
                type="submit" 
                className="submit-btn-minimal"
                disabled={!selectedCategory || !selectedDate}
                whileHover={{ scale: selectedCategory && selectedDate ? 1.02 : 1 }}
                whileTap={{ scale: selectedCategory && selectedDate ? 0.98 : 1 }}
              >
                <span>Оформить</span>
                <ArrowRight size={16} strokeWidth={1.5} />
              </motion.button>
            </form>

            {isSubmitted && (
              <motion.div 
                className="success-minimal"
                initial={{ opacity: 0, scale: 0.9 }}
                animate={{ opacity: 1, scale: 1 }}
              >
                <Check size={24} strokeWidth={2} />
                <span>Аренда оформлена</span>
              </motion.div>
            )}
          </motion.section>
        </div>

        <ImageSlider />

        <section className="features-minimal-section">
          <motion.div 
            className="features-list-minimal"
            initial={{ opacity: 0 }}
            whileInView={{ opacity: 1 }}
            viewport={{ once: true }}
            transition={{ duration: 0.6 }}
          >
            {[
              { title: 'Премиум оборудование', text: 'Только профессиональные инструменты' },
              { title: 'Быстрая доставка', text: 'Доставка в день заказа' },
              { title: 'Техподдержка', text: 'Круглосуточная поддержка' }
            ].map((feature, index) => (
              <motion.div 
                key={index}
                className="feature-item-minimal"
                initial={{ opacity: 0, y: 20 }}
                whileInView={{ opacity: 1, y: 0 }}
                viewport={{ once: true }}
                transition={{ delay: index * 0.1 }}
              >
                <div className="feature-line-minimal"></div>
                <div className="feature-content-minimal">
                  <h3 className="feature-title-minimal">{feature.title}</h3>
                  <p className="feature-text-minimal">{feature.text}</p>
                </div>
              </motion.div>
            ))}
          </motion.div>
        </section>
      </div>
    </div>
  )
}

export default Rent
